<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Support\Str;
use Stichoza\GoogleTranslate\GoogleTranslate; // Added
use Exception; // Added

class Actualite extends Model
{
    use HasFactory;

    protected $fillable = [
        'titre',
        'titre_en', // Added
        'slug',
        'description',
        'description_en', // Added
        'image',
        'datePublication',
    ];

    // Relation hasOne for a unique detail
    public function details()
    {
        return $this->hasOne(ActualiteDetail::class, 'actualite_id');
    }

    protected static function boot()
    {
        parent::boot();

        static::creating(function ($actualite) {
            $actualite->slug = Str::slug($actualite->titre);
            $count = static::where('slug', 'LIKE', $actualite->slug . '%')->count();
            $actualite->slug = $count ? $actualite->slug . '-' . ($count + 1) : $actualite->slug;
            // Auto-translate titre and description to English
            try {
                $translator = new GoogleTranslate();
                $translator->setSource('fr');
                $translator->setTarget('en');

                if (!$actualite->titre_en && $actualite->titre) {
                    $actualite->titre_en = $translator->translate($actualite->titre);
                }
                if (!$actualite->description_en && $actualite->description) {
                    $actualite->description_en = $translator->translate($actualite->description);
                }

            } catch (Exception $e) {
                // Log the error or handle it gracefully
                error_log("Translation error: " . $e->getMessage());
                // Optionally, set default values
                $actualite->titre_en = 'Translation failed';
                $actualite->description_en = 'Translation failed';
            }

        });

        static::updating(function ($actualite) {
            if ($actualite->isDirty('titre')) {
                $actualite->slug = Str::slug($actualite->titre);
                $count = static::where('slug', 'LIKE', $actualite->slug . '%')->where('id', '!=', $actualite->id)->count();
                $actualite->slug = $count ? $actualite->slug . '-' . ($count + 1) : $actualite->slug;

                // Auto-translate titre and description to English on update
                 try {
                    $translator = new GoogleTranslate();
                    $translator->setSource('fr');
                    $translator->setTarget('en');

                    if ($actualite->isDirty('titre') && $actualite->titre) {
                        $actualite->titre_en = $translator->translate($actualite->titre);
                    }
                    if ($actualite->isDirty('description') && $actualite->description) {
                        $actualite->description_en = $translator->translate($actualite->description);
                    }
                 }  catch (Exception $e) {
                    // Log the error or handle it gracefully
                    error_log("Translation error: " . $e->getMessage());
                    // Optionally, set default values
                    $actualite->titre_en = 'Translation failed';
                    $actualite->description_en = 'Translation failed';
                }
            }
        });
    }

    // Use the slug as the route key
    public function getRouteKeyName()
    {
        return 'slug';
    }
}