<?php

namespace App\Http\Controllers\API;

use App\Http\Controllers\Controller;
use App\Models\Ville;
use App\Models\Pays;
use Illuminate\Http\Request;
use Illuminate\Http\Response;

class VilleController extends Controller
{
    /**
     * Display a listing of the resource without pagination.
     */
    public function index(Request $request)
    {
        $query = $request->input('query');
        $villesQuery = Ville::with('pays');

        if ($query) {
            $villesQuery->where('nom', 'LIKE', '%' . $query . '%');
        }

        $villes = $villesQuery->get(); // Utilisation de get() pour récupérer tous les résultats sans pagination

        if ($villes->isEmpty()) {
            return response()->json(['message' => 'Aucune ville trouvée'], Response::HTTP_NOT_FOUND);
        }

        return response()->json($villes, Response::HTTP_OK);
    }

    /**
     * Récupérer les villes par pays sans pagination.
     */
    public function getVillesByPays($paysId)
    {
        if (!Pays::find($paysId)) {
            return response()->json(['message' => 'Pays non trouvé'], Response::HTTP_NOT_FOUND);
        }

        $villes = Ville::where('pays_id', $paysId)->with('pays')->get();
        return response()->json($villes);
    }

    /**
     * Store a newly created resource in storage.
     */
    public function store(Request $request)
    {
        $request->validate([
            'nom' => 'required|string|max:255',
            'pays_id' => 'required|exists:pays,id',
        ]);

        $ville = Ville::create([
            'nom' => $request->nom,
            'pays_id' => $request->pays_id,
        ]);

        return response()->json([
            'message' => 'Ville créée avec succès',
            'ville' => $ville
        ], Response::HTTP_CREATED);
    }

    /**
     * Display the specified resource.
     */
    public function show(string $id)
    {
        $ville = Ville::with('pays')->find($id);

        if (!$ville) {
            return response()->json(['message' => 'Ville non trouvée'], Response::HTTP_NOT_FOUND);
        }

        return response()->json($ville, Response::HTTP_OK);
    }

    /**
     * Update the specified resource in storage.
     */
    public function update(Request $request, string $id)
    {
        $request->validate([
            'nom' => 'string|max:255',
            'pays_id' => 'exists:pays,id',
        ]);

        $ville = Ville::find($id);

        if (!$ville) {
            return response()->json(['message' => 'Ville non trouvée'], Response::HTTP_NOT_FOUND);
        }

        $ville->update($request->all());

        return response()->json([
            'message' => 'Ville mise à jour avec succès',
            'ville' => $ville
        ], Response::HTTP_OK);
    }

    /**
     * Remove the specified resource from storage.
     */
    public function destroy(string $id)
    {
        $ville = Ville::find($id);

        if (!$ville) {
            return response()->json(['message' => 'Ville non trouvée'], Response::HTTP_NOT_FOUND);
        }

        $ville->delete();

        return response()->json(['message' => 'Ville supprimée avec succès'], Response::HTTP_OK);
    }
}
