<?php

namespace App\Http\Controllers\API;

use App\Http\Controllers\Controller;
use App\Models\Video;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Validator;
use Stichoza\GoogleTranslate\GoogleTranslate;

class VideoController extends Controller
{
    public function index(Request $request)
    {
        $lang = $request->header('Accept-Language', 'fr');
        
        $videos = Video::all()->map(function ($video) use ($lang) {
            return [
                'id' => $video->id,
                'title' => $lang === 'en' ? $video->title_en : $video->title,
                'youtube_id' => $video->youtube_id,
                'date' => $video->date,
                'created_at' => $video->created_at,
                'updated_at' => $video->updated_at,
            ];
        });

        return response()->json(['data' => $videos], 200);
    }

    public function store(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'title' => 'required|string|max:255',
            'youtube_id' => 'required|string|max:255',
            'date' => 'nullable|string|max:255',
        ]);

        if ($validator->fails()) {
            return response()->json(['errors' => $validator->errors()], 422);
        }

        try {
            $translator = new GoogleTranslate();
            $translator->setSource('fr');
            $translator->setTarget('en');

            $video = Video::create([
                'title' => $request->title,
                'title_en' => $translator->translate($request->title),
                'youtube_id' => $request->youtube_id,
                'date' => $request->date,
            ]);

            return response()->json([
                'data' => $video,
                'message' => 'Video created successfully'
            ], 201);
        } catch (\Exception $e) {
            return response()->json([
                'error' => 'Error creating video: ' . $e->getMessage()
            ], 500);
        }
    }

    public function show(Request $request, Video $video)
    {
        $lang = $request->header('Accept-Language', 'fr');
        
        $videoData = [
            'id' => $video->id,
            'title' => $lang === 'en' ? $video->title_en : $video->title,
            'youtube_id' => $video->youtube_id,
            'date' => $video->date,
            'created_at' => $video->created_at,
            'updated_at' => $video->updated_at,
        ];

        return response()->json(['data' => $videoData], 200);
    }

    public function update(Request $request, Video $video)
    {
        $validator = Validator::make($request->all(), [
            'title' => 'required|string|max:255',
            'youtube_id' => 'required|string|max:255',
            'date' => 'nullable|string|max:255',
        ]);

        if ($validator->fails()) {
            return response()->json(['errors' => $validator->errors()], 422);
        }

        try {
            $translator = new GoogleTranslate();
            $translator->setSource('fr');
            $translator->setTarget('en');

            $video->update([
                'title' => $request->title,
                'title_en' => $translator->translate($request->title),
                'youtube_id' => $request->youtube_id,
                'date' => $request->date,
            ]);

            return response()->json([
                'data' => $video,
                'message' => 'Video updated successfully'
            ], 200);
        } catch (\Exception $e) {
            return response()->json([
                'error' => 'Error updating video: ' . $e->getMessage()
            ], 500);
        }
    }

    public function destroy(Video $video)
    {
        try {
            $video->delete();
            return response()->json(['message' => 'Video deleted successfully'], 204);
        } catch (\Exception $e) {
            return response()->json([
                'error' => 'Error deleting video: ' . $e->getMessage()
            ], 500);
        }
    }
}