<?php
namespace App\Http\Controllers\API;

use App\Http\Controllers\Controller;
use Illuminate\Http\Request;
use App\Models\Stat;
use Stichoza\GoogleTranslate\GoogleTranslate;
use Illuminate\Support\Facades\Log;

class StatController extends Controller
{
    public function index(Request $request)
    {
        try {
            $lang = $request->header('Accept-Language', 'fr');

            $stats = Stat::all()->map(function ($stat) use ($lang) {
                return [
                    'id' => $stat->id,
                    'label' => $lang === 'en' ? $stat->label_en : $stat->label,
                    'value' => $stat->value,
                    'created_at' => $stat->created_at,
                ];
            });

            return response()->json($stats, 200);
        } catch (\Exception $e) {
            Log::error("Erreur lors de la récupération des stats : " . $e->getMessage());
            return response()->json(['error' => 'Une erreur est survenue lors de la récupération des stats.'], 500);
        }
    }

    public function store(Request $request)
    {
        try {
            $validated = $request->validate([
                'label' => 'required|string|max:255',
                'value' => 'nullable|integer',
            ]);

            // Auto-translate to English
            $translator = new GoogleTranslate();
            $translator->setSource('fr');
            $translator->setTarget('en');
            $label_en = $translator->translate($validated['label']);

            $stat = Stat::create([
                'label' => $validated['label'],
                'label_en' => $label_en,
                'value' => $validated['value'],
            ]);

            return response()->json($stat, 201);
        } catch (\Exception $e) {
            Log::error("Erreur lors de la création de la stat : " . $e->getMessage());
            return response()->json(['error' => 'Impossible d\'ajouter la stat.'], 500);
        }
    }

    public function show(Request $request, $id)
    {
        try {
            $lang = $request->header('Accept-Language', 'fr');
            
            $stat = Stat::find($id);
            if (!$stat) {
                return response()->json(['error' => 'Stat introuvable.'], 404);
            }

            return response()->json([
                'id' => $stat->id,
                'label' => $lang === 'en' ? $stat->label_en : $stat->label,
                'value' => $stat->value,
                'created_at' => $stat->created_at,
            ], 200);
        } catch (\Exception $e) {
            Log::error("Erreur lors de la récupération de la stat : " . $e->getMessage());
            return response()->json(['error' => 'Une erreur est survenue.'], 500);
        }
    }

    public function update(Request $request, $id)
    {
        try {
            $validated = $request->validate([
                'label' => 'required|string|max:255',
                'value' => 'nullable|integer',
            ]);

            $stat = Stat::find($id);
            if (!$stat) {
                return response()->json(['error' => 'Stat introuvable.'], 404);
            }

            // Update and re-translate label
            $translator = new GoogleTranslate();
            $translator->setSource('fr');
            $translator->setTarget('en');
            $label_en = $translator->translate($validated['label']);

            $stat->update([
                'label' => $validated['label'],
                'label_en' => $label_en,
                'value' => $validated['value'],
            ]);

            return response()->json($stat, 200);
        } catch (\Exception $e) {
            Log::error("Erreur lors de la mise à jour de la stat : " . $e->getMessage());
            return response()->json(['error' => 'Impossible de mettre à jour la stat.'], 500);
        }
    }

    public function destroy($id)
    {
        try {
            $stat = Stat::find($id);
            if (!$stat) {
                return response()->json(['error' => 'Stat introuvable.'], 404);
            }

            $stat->delete();
            return response()->json(['message' => 'Stat supprimée avec succès.'], 200);
        } catch (\Exception $e) {
            Log::error("Erreur lors de la suppression de la stat : " . $e->getMessage());
            return response()->json(['error' => 'Impossible de supprimer la stat.'], 500);
        }
    }
}