<?php

namespace App\Http\Controllers\API;

use App\Http\Controllers\Controller;
use Illuminate\Http\Request;
use App\Models\Sponsor;
use Illuminate\Support\Facades\Storage;
use Illuminate\Support\Facades\Validator;
use Illuminate\Database\Eloquent\ModelNotFoundException;
use Stichoza\GoogleTranslate\GoogleTranslate;
use Exception;

class SponsorController extends Controller
{
    private const IMAGE_STORAGE_PATH = 'public/images_sponsor';
    private const MAX_IMAGE_SIZE = 10240; // 10 MB (10240 KB)

    /**
     * Display a listing of the sponsors.
     *
     * @param  Request  $request
     * @return \Illuminate\Http\JsonResponse
     */
    public function index(Request $request)
    {
        $lang = $request->header('Accept-Language', 'fr');

        $sponsors = Sponsor::all()->map(function ($sponsor) use ($lang) {
            $sponsor->nom = $lang === 'en' ? $sponsor->nom_en : $sponsor->nom;
            $sponsor->description = $lang === 'en' ? $sponsor->description_en : $sponsor->description;
            $sponsor->image = $sponsor->image ? url('/api/images_sponsor/' . basename($sponsor->image)) : null;
            return $sponsor;
        });

        return response()->json($sponsors);
    }

    /**
     * Display the specified sponsor.
     *
     * @param  Request  $request
     * @param  int  $id
     * @return \Illuminate\Http\JsonResponse
     */
    public function show(Request $request, $id)
    {
        $lang = $request->header('Accept-Language', 'fr');

        try {
            $sponsor = Sponsor::findOrFail($id);
        } catch (ModelNotFoundException $e) {
            return response()->json([
                'status' => false,
                'message' => 'Sponsor non trouvé'
            ], 404);
        }

        $sponsor->nom = $lang === 'en' ? $sponsor->nom_en : $sponsor->nom;
        $sponsor->description = $lang === 'en' ? $sponsor->description_en : $sponsor->description;
        $sponsor->image = $sponsor->image ? url('/api/images_sponsor/' . basename($sponsor->image)) : null;

        return response()->json($sponsor);
    }

    /**
     * Store a newly created sponsor in storage.
     *
     * @param  Request  $request
     * @return \Illuminate\Http\JsonResponse
     */
    public function store(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'nom' => 'required|string|max:255',
            'description' => 'nullable|string',
            'image' => ['nullable', 'image', 'mimes:jpeg,png,jpg,gif', 'max:' . self::MAX_IMAGE_SIZE],
        ]);

        if ($validator->fails()) {
            return response()->json([
                'status' => false,
                'errors' => $validator->errors()
            ], 422);
        }

        try {
            $sponsorData = $request->except('image');

            // Auto-translate nom and description to English
            $translator = new GoogleTranslate();
            $translator->setSource('fr');
            $translator->setTarget('en');
            $sponsorData['nom_en'] = $translator->translate($request->nom);
            $sponsorData['description_en'] = $request->description ? $translator->translate($request->description) : null;

            if ($request->hasFile('image')) {
                $imagePath = $request->file('image')->store(self::IMAGE_STORAGE_PATH);
                $sponsorData['image'] = $imagePath;
            }

            $sponsor = Sponsor::create($sponsorData);

            $sponsor->image = $sponsor->image ? url('/api/images_sponsor/' . basename($sponsor->image)) : null;

            return response()->json([
                'status' => true,
                'message' => 'Sponsor créé avec succès',
                'data' => $sponsor
            ], 201);
        } catch (Exception $e) {
            return response()->json([
                'status' => false,
                'message' => 'Erreur lors de la création du sponsor: ' . $e->getMessage()
            ], 500);
        }
    }

    /**
     * Update the specified sponsor in storage.
     *
     * @param  Request  $request
     * @param  int  $id
     * @return \Illuminate\Http\JsonResponse
     */
    public function update(Request $request, $id)
    {
        $validator = Validator::make($request->all(), [
            'nom' => 'sometimes|required|string|max:255',
            'description' => 'nullable|string',
            'image' => ['nullable', 'image', 'mimes:jpeg,png,jpg,gif', 'max:' . self::MAX_IMAGE_SIZE],
        ]);

        if ($validator->fails()) {
            return response()->json([
                'status' => false,
                'errors' => $validator->errors()
            ], 422);
        }

        try {
            $sponsor = Sponsor::findOrFail($id);
            $sponsorData = $request->except('image');

            // Auto-translate nom and description to English if updated
            $translator = new GoogleTranslate();
            $translator->setSource('fr');
            $translator->setTarget('en');

            if ($request->has('nom')) {
                $sponsorData['nom_en'] = $translator->translate($request->nom);
            }

            if ($request->has('description')) {
                $sponsorData['description_en'] = $translator->translate($request->description);
            }

            if ($request->hasFile('image')) {
                if ($sponsor->image) {
                    Storage::delete($sponsor->image);
                }
                $imagePath = $request->file('image')->store(self::IMAGE_STORAGE_PATH);
                $sponsorData['image'] = $imagePath;
            }

            $sponsor->update($sponsorData);

            $sponsor->image = $sponsor->image ? url('/api/images_sponsor/' . basename($sponsor->image)) : null;

            return response()->json([
                'status' => true,
                'message' => 'Sponsor mis à jour avec succès',
                'data' => $sponsor
            ], 200);
        } catch (ModelNotFoundException $e) {
            return response()->json([
                'status' => false,
                'message' => 'Sponsor non trouvé'
            ], 404);
        } catch (Exception $e) {
            return response()->json([
                'status' => false,
                'message' => 'Erreur lors de la mise à jour du sponsor: ' . $e->getMessage()
            ], 500);
        }
    }

    /**
     * Remove the specified sponsor from storage.
     *
     * @param  int  $id
     * @return \Illuminate\Http\JsonResponse
     */
    public function destroy($id)
    {
        try {
            $sponsor = Sponsor::findOrFail($id);
        } catch (ModelNotFoundException $e) {
            return response()->json([
                'status' => false,
                'message' => 'Sponsor non trouvé'
            ], 404);
        }

        try {
            if ($sponsor->image) {
                Storage::delete($sponsor->image);
            }

            $sponsor->delete();

            return response()->json([
                'status' => true,
                'message' => 'Sponsor supprimé avec succès'
            ], 200);
        } catch (Exception $e) {
            return response()->json([
                'status' => false,
                'message' => 'Erreur lors de la suppression du sponsor: ' . $e->getMessage()
            ], 500);
        }
    }
}