<?php

namespace App\Http\Controllers\API;

use App\Http\Controllers\Controller;
use App\Models\Slide;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Storage;
use Stichoza\GoogleTranslate\GoogleTranslate;
use Illuminate\Support\Facades\Log;

class SlideController extends Controller
{
    private const IMAGE_STORAGE_PATH = 'public/images_slides'; // Nouveau chemin spécifique aux slides

    private const MAX_IMAGE_SIZE = 30720; // 30 MB (30720 KB)

    /**
     * Display a listing of the resource.
     */
    public function index(Request $request)
    {
        try {
            $lang = $request->header('Accept-Language', 'fr'); // Langue demandée (FR par défaut)

            $slides = Slide::orderBy('order')->get()->map(function ($slide) use ($lang) {
                return [
                    'id' => $slide->id,
                    'image' => $slide->image ? url('/api/images_slides/' . basename($slide->image)) : null,
                    'title' => $lang === 'en' ? $slide->title_en : $slide->title,
                    'description' => $lang === 'en' ? $slide->description_en : $slide->description,
                    'order' => $slide->order,
                    'created_at' => $slide->created_at,
                ];
            });

            return response()->json($slides);
        } catch (\Exception $e) {
            Log::error("Erreur lors de la récupération des slides : " . $e->getMessage());
            return response()->json(['error' => 'Une erreur est survenue lors de la récupération des slides.'], 500);
        }
    }

    /**
     * Store a newly created resource in storage.
     */
    public function store(Request $request)
    {
        try {
            $request->validate([
                'title' => 'required|string',
                'description' => 'nullable|string',
                'image' => ['required', 'image', 'mimes:jpeg,png,jpg,gif', 'max:' . self::MAX_IMAGE_SIZE],
                'order' => 'nullable|integer',
            ]);

            $imagePath = $request->file('image')->store(self::IMAGE_STORAGE_PATH);

            // Translate title and description to English
            $translator = new GoogleTranslate();
            $translator->setSource('fr');
            $translator->setTarget('en');

            $title_en = $translator->translate($request->input('title'));
            $description_en = $translator->translate($request->input('description'));

            $slide = Slide::create([
                'title' => $request->input('title'),
                'description' => $request->input('description'),
                'title_en' => $title_en,
                'description_en' => $description_en,
                'image' => $imagePath,
                'order' => $request->input('order', 0)
            ]);

            // Add URL for the image
            $slide->image = url('/api/images_slides/' . basename($slide->image));

            return response()->json($slide, 201);

        } catch (\Exception $e) {
            Log::error("Erreur lors de la création du slide : " . $e->getMessage());
            return response()->json(['error' => 'Impossible d\'ajouter le slide.'], 500);
        }
    }

    /**
     * Display the specified resource.
     */
    public function show(Request $request, Slide $slide)
    {
        try {
            $lang = $request->header('Accept-Language', 'fr'); // Langue demandée

            return response()->json([
                'id' => $slide->id,
                'image' => $slide->image ? url('/api/images_slides/' . basename($slide->image)) : null,
                'title' => $lang === 'en' ? $slide->title_en : $slide->title,
                'description' => $lang === 'en' ? $slide->description_en : $slide->description,
                'order' => $slide->order,
                'created_at' => $slide->created_at,
            ], 200);

        } catch (\Exception $e) {
            Log::error("Erreur lors de la récupération du slide : " . $e->getMessage());
            return response()->json(['error' => 'Une erreur est survenue.'], 500);
        }
    }

    /**
     * Update the specified resource in storage.
     */
    public function update(Request $request, Slide $slide)
    {
        try {
            $request->validate([
                'title' => 'sometimes|required|string',
                'description' => 'nullable|string',
                'image' => ['nullable', 'image', 'mimes:jpeg,png,jpg,gif', 'max:' . self::MAX_IMAGE_SIZE],
                'order' => 'nullable|integer',
            ]);


            // Translate title and description to English
            $translator = new GoogleTranslate();
            $translator->setSource('fr');
            $translator->setTarget('en');

            if ($request->has('title')) {
                $slide->title = $request->input('title');
                $slide->title_en = $translator->translate($request->input('title'));
            }

            if ($request->has('description')) {
                $slide->description = $request->input('description');
                $slide->description_en = $translator->translate($request->input('description'));
            }

            if ($request->has('order')) {
                $slide->order = $request->input('order');
            }


            if ($request->hasFile('image')) {
                if ($slide->image) {
                    Storage::delete($slide->image);
                }
                $slide->image = $request->file('image')->store(self::IMAGE_STORAGE_PATH);
            }

            $slide->save();

            // Add URL for the image
            $slide->image = $slide->image ? url('/api/images_slides/' . basename($slide->image)) : null;

            return response()->json($slide);
        } catch (\Exception $e) {
            Log::error("Erreur lors de la mise à jour du slide : " . $e->getMessage());
            return response()->json(['error' => 'Impossible de mettre à jour le slide.'], 500);
        }
    }

    /**
     * Remove the specified resource from storage.
     */
    public function destroy(Slide $slide)
    {
        try {
            if ($slide->image) {
                Storage::delete($slide->image);
            }

            $slide->delete();

            return response()->json(null, 204);

        } catch (\Exception $e) {
            Log::error("Erreur lors de la suppression du slide : " . $e->getMessage());
            return response()->json(['error' => 'Impossible de supprimer le slide.'], 500);
        }
    }
}