<?php

namespace App\Http\Controllers\API;

use App\Http\Controllers\Controller;
use Illuminate\Http\Request;
use App\Models\Reglementation;
use Illuminate\Support\Facades\Storage;

class ReglementationController extends Controller
{
    private const IMAGE_STORAGE_PATH = 'public/images';
    private const DOCUMENT_STORAGE_PATH = 'public/documents';

    private const MAX_IMAGE_SIZE = 3072000000000; // 30 MB (30720 KB)
    private const MAX_DOCUMENT_SIZE = 25600000000000000; // 25 MB (25600 KB)

    /**
     * Display a listing of the resource.
     */
    public function index(Request $request)
    {
        $lang = $request->header('Accept-Language', 'fr'); // Langue demandée (FR par défaut)

        $reglementations = Reglementation::all()->map(function ($reglementation) use ($lang) {
            $reglementation->title = $lang === 'en' ? $reglementation->title_en : $reglementation->title;
            $reglementation->description = $lang === 'en' ? $reglementation->description_en : $reglementation->description;
            $reglementation->image = $reglementation->image ? url('/api/images_reglementation/' . basename($reglementation->image)) : null;
            $reglementation->document = $reglementation->document ? url('/api/documents_reglementation/' . basename($reglementation->document)) : null;
            return $reglementation;
        });

        return response()->json($reglementations);
    }

    /**
     * Store a newly created resource in storage.
     */
    public function store(Request $request)
    {
        $request->validate([
            'title' => 'required|string',
            'description' => 'nullable|string',
            'image' => ['nullable', 'image', 'mimes:jpeg,png,jpg,gif', 'max:' . self::MAX_IMAGE_SIZE],
            'document' => ['required', 'file', 'mimes:pdf,doc,docx', 'max:' . self::MAX_DOCUMENT_SIZE],
        ]);

        $documentPath = null;
        if ($request->hasFile('document')) {
            $documentPath = $request->file('document')->store(self::DOCUMENT_STORAGE_PATH);
        }

        $imagePath = null;
        if ($request->hasFile('image')) {
            $imagePath = $request->file('image')->store(self::IMAGE_STORAGE_PATH);
        }

        $reglementation = Reglementation::create([
            'title' => $request->input('title'),
            'description' => $request->input('description'),
            'image' => $imagePath,
            'document' => $documentPath,
        ]);

        // Ajouter les URLs pour les fichiers
        $reglementation->image = $reglementation->image ? url('/api/images_reglementation/' . basename($reglementation->image)) : null;
        $reglementation->document = $reglementation->document ? url('/api/documents_reglementation/' . basename($reglementation->document)) : null;

        return response()->json($reglementation, 201);
    }

    /**
     * Display the specified resource.
     */
    public function show(Request $request, Reglementation $reglementation)
    {
        $lang = $request->header('Accept-Language', 'fr'); // Langue demandée

        $reglementation->title = $lang === 'en' ? $reglementation->title_en : $reglementation->title;
        $reglementation->description = $lang === 'en' ? $reglementation->description_en : $reglementation->description;
        $reglementation->image = $reglementation->image ? url('/api/images_reglementation/' . basename($reglementation->image)) : null;
        $reglementation->document = $reglementation->document ? url('/api/documents_reglementation/' . basename($reglementation->document)) : null;

        return response()->json($reglementation);
    }

    /**
     * Update the specified resource in storage.
     */
    public function update(Request $request, Reglementation $reglementation)
    {
        $request->validate([
            'title' => 'sometimes|required|string',
            'description' => 'nullable|string',
            'image' => ['nullable', 'image', 'mimes:jpeg,png,jpg,gif', 'max:' . self::MAX_IMAGE_SIZE],
            'document' => ['nullable', 'file', 'mimes:pdf,doc,docx', 'max:' . self::MAX_DOCUMENT_SIZE],
        ]);

        if ($request->has('title')) {
            $reglementation->title = $request->input('title');
        }

        if ($request->has('description')) {
            $reglementation->description = $request->input('description');
        }

        if ($request->hasFile('document')) {
            if ($reglementation->document) {
                Storage::delete($reglementation->document);
            }
            $reglementation->document = $request->file('document')->store(self::DOCUMENT_STORAGE_PATH);
        }

        if ($request->hasFile('image')) {
            if ($reglementation->image) {
                Storage::delete($reglementation->image);
            }
            $reglementation->image = $request->file('image')->store(self::IMAGE_STORAGE_PATH);
        }

        $reglementation->save();

        // Ajouter les URLs pour les fichiers
        $reglementation->image = $reglementation->image ? url('/api/images_reglementation/' . basename($reglementation->image)) : null;
        $reglementation->document = $reglementation->document ? url('/api/documents_reglementation/' . basename($reglementation->document)) : null;

        return response()->json($reglementation);
    }

    /**
     * Remove the specified resource from storage.
     */
    public function destroy(Reglementation $reglementation)
    {
        if ($reglementation->document) {
            Storage::delete($reglementation->document);
        }
        if ($reglementation->image) {
            Storage::delete($reglementation->image);
        }

        $reglementation->delete();

        return response()->json(null, 204);
    }
}
