<?php

namespace App\Http\Controllers\API;

use App\Http\Controllers\Controller;
use Illuminate\Http\Request;
use App\Models\Publication;
use Illuminate\Support\Facades\Storage;
use Stichoza\GoogleTranslate\GoogleTranslate;
class PublicationController extends Controller
{
    private const PHOTO_STORAGE_PATH = 'public/photos';
    private const DOCUMENT_STORAGE_PATH = 'public/documents';

    private const MAX_PHOTO_SIZE = 30720; // 30 MB (30720 KB)
    private const MAX_DOCUMENT_SIZE = 25600; // 25 MB (25600 KB)

    /**
     * Display a listing of the resource.
     */
    public function index(Request $request)
    {
        $lang = $request->header('Accept-Language', 'fr'); // Langue demandée (FR par défaut)

        $publications = Publication::all()->map(function ($publication) use ($lang) {
            $publication->title = $lang === 'en' ? $publication->title_en : $publication->title;
            $publication->description = $lang === 'en' ? $publication->description_en : $publication->description;
            $publication->photo = $publication->photo ? url('/api/images_publication/' . basename($publication->photo)) : null;
            $publication->document = $publication->document ? url('/api/documents_publication/' . basename($publication->document)) : null;
            return $publication;
        });

        return response()->json($publications);
    }

    /**
     * Store a newly created resource in storage.
     */
    public function store(Request $request)
    {
        $request->validate([
            'title' => 'required|string',
            'description' => 'nullable|string',
            'photo' => ['nullable', 'image', 'mimes:jpeg,png,jpg,gif', 'max:' . self::MAX_PHOTO_SIZE],
            'document' => ['required', 'file', 'mimes:pdf,doc,docx', 'max:' . self::MAX_DOCUMENT_SIZE],
            'publication_date' => 'nullable|date',
        ]);

        $documentPath = $request->hasFile('document') ? $request->file('document')->store(self::DOCUMENT_STORAGE_PATH) : null;
        $photoPath = $request->hasFile('photo') ? $request->file('photo')->store(self::PHOTO_STORAGE_PATH) : null;

        $publication = Publication::create([
            'title' => $request->input('title'),
            'description' => $request->input('description'),
            'photo' => $photoPath,
            'document' => $documentPath,
            'publication_date' => $request->input('publication_date'),
        ]);

        $publication->photo = $publication->photo ? url('/api/images_publication/' . basename($publication->photo)) : null;
        $publication->document = $publication->document ? url('/api/documents_publication/' . basename($publication->document)) : null;

        return response()->json($publication, 201);
    }

    /**
     * Display the specified resource.
     */
    public function show(Request $request, Publication $publication)
    {
        $lang = $request->header('Accept-Language', 'fr'); // Langue demandée

        $publication->title = $lang === 'en' ? $publication->title_en : $publication->title;
        $publication->description = $lang === 'en' ? $publication->description_en : $publication->description;
        $publication->photo = $publication->photo ? url('/api/images_publication/' . basename($publication->photo)) : null;
        $publication->document = $publication->document ? url('/api/documents_publication/' . basename($publication->document)) : null;

        return response()->json($publication);
    }

    /**
     * Update the specified resource in storage.
     */
    public function update(Request $request, Publication $publication)
    {
        $request->validate([
            'title' => 'sometimes|required|string',
            'description' => 'nullable|string',
            'photo' => ['nullable', 'image', 'mimes:jpeg,png,jpg,gif', 'max:' . self::MAX_PHOTO_SIZE],
            'document' => ['nullable', 'file', 'mimes:pdf,doc,docx', 'max:' . self::MAX_DOCUMENT_SIZE],
            'publication_date' => 'nullable|date',
        ]);

        if ($request->has('title')) {
            $publication->title = $request->input('title');
        }

        if ($request->has('description')) {
            $publication->description = $request->input('description');
        }

        if ($request->has('publication_date')) {
            $publication->publication_date = $request->input('publication_date');
        }

        if ($request->hasFile('document')) {
            if ($publication->document) {
                Storage::delete($publication->document);
            }
            $publication->document = $request->file('document')->store(self::DOCUMENT_STORAGE_PATH);
        }

        if ($request->hasFile('photo')) {
            if ($publication->photo) {
                Storage::delete($publication->photo);
            }
            $publication->photo = $request->file('photo')->store(self::PHOTO_STORAGE_PATH);
        }

        $publication->save();

        $publication->photo = $publication->photo ? url('/api/images_publication/' . basename($publication->photo)) : null;
        $publication->document = $publication->document ? url('/api/documents_publication/' . basename($publication->document)) : null;

        return response()->json($publication);
    }

    /**
     * Remove the specified resource from storage.
     */
    public function destroy(Publication $publication)
    {
        if ($publication->document) {
            Storage::delete($publication->document);
        }
        if ($publication->photo) {
            Storage::delete($publication->photo);
        }

        $publication->delete();

        return response()->json(null, 204);
    }
}
