<?php
namespace App\Http\Controllers\API;

use App\Http\Controllers\Controller;
use Illuminate\Http\Request;
use App\Models\Presentation;
use Illuminate\Http\Response;
use Illuminate\Support\Facades\Validator;
use Stichoza\GoogleTranslate\GoogleTranslate;
use Illuminate\Support\Facades\Log;

class PresentationController extends Controller
{
    /**
     * Liste toutes les présentations avec la langue sélectionnée
     */
    public function index(Request $request)
    {
        try {
            $lang = $request->header('Accept-Language', 'fr'); // Langue demandée

            $presentations = Presentation::all()->map(function ($presentation) use ($lang) {
                return [
                    'id' => $presentation->id,
                    'description' => $lang === 'en' ? $presentation->description_en : $presentation->description,
                    'created_at' => $presentation->created_at,
                ];
            });

            return response()->json(['data' => $presentations], Response::HTTP_OK);
        } catch (\Exception $e) {
            Log::error("Erreur lors de la récupération des présentations : " . $e->getMessage());
            return response()->json(['error' => 'Une erreur est survenue.'], Response::HTTP_INTERNAL_SERVER_ERROR);
        }
    }

    /**
     * Crée une nouvelle présentation avec traduction automatique
     */
    public function store(Request $request)
    {
        try {
            $validator = Validator::make($request->all(), [
                'description' => 'required|string',
            ]);

            if ($validator->fails()) {
                return response()->json(['errors' => $validator->errors()], Response::HTTP_UNPROCESSABLE_ENTITY);
            }

            // Traduction automatique
            $translator = new GoogleTranslate();
            $translator->setSource('fr');
            $translator->setTarget('en');
            $description_en = $translator->translate($request->description);

            $presentation = Presentation::create([
                'description' => $request->description,
                'description_en' => $description_en
            ]);

            return response()->json(['data' => $presentation, 'message' => 'Présentation créée avec succès'], Response::HTTP_CREATED);
        } catch (\Exception $e) {
            Log::error("Erreur lors de la création de la présentation : " . $e->getMessage());
            return response()->json(['error' => 'Impossible d\'ajouter la présentation.'], Response::HTTP_INTERNAL_SERVER_ERROR);
        }
    }

    /**
     * Affiche une présentation spécifique
     */
    public function show(Request $request, $id)
    {
        try {
            $lang = $request->header('Accept-Language', 'fr');
            $presentation = Presentation::find($id);

            if (!$presentation) {
                return response()->json(['error' => 'Présentation introuvable.'], Response::HTTP_NOT_FOUND);
            }

            return response()->json([
                'id' => $presentation->id,
                'description' => $lang === 'en' ? $presentation->description_en : $presentation->description,
                'created_at' => $presentation->created_at,
            ], Response::HTTP_OK);
        } catch (\Exception $e) {
            Log::error("Erreur lors de la récupération de la présentation : " . $e->getMessage());
            return response()->json(['error' => 'Une erreur est survenue.'], Response::HTTP_INTERNAL_SERVER_ERROR);
        }
    }

    /**
     * Met à jour une présentation
     */
    public function update(Request $request, $id)
    {
        try {
            $presentation = Presentation::find($id);
            if (!$presentation) {
                return response()->json(['error' => 'Présentation introuvable.'], Response::HTTP_NOT_FOUND);
            }

            $validator = Validator::make($request->all(), [
                'description' => 'required|string',
            ]);

            if ($validator->fails()) {
                return response()->json(['errors' => $validator->errors()], Response::HTTP_UNPROCESSABLE_ENTITY);
            }

            $translator = new GoogleTranslate();
            $translator->setSource('fr');
            $translator->setTarget('en');
            $description_en = $translator->translate($request->description);

            $presentation->update([
                'description' => $request->description,
                'description_en' => $description_en
            ]);

            return response()->json(['data' => $presentation, 'message' => 'Présentation mise à jour avec succès'], Response::HTTP_OK);
        } catch (\Exception $e) {
            Log::error("Erreur lors de la mise à jour de la présentation : " . $e->getMessage());
            return response()->json(['error' => 'Impossible de mettre à jour la présentation.'], Response::HTTP_INTERNAL_SERVER_ERROR);
        }
    }

    /**
     * Supprime une présentation
     */
    public function destroy($id)
    {
        try {
            $presentation = Presentation::find($id);
            if (!$presentation) {
                return response()->json(['error' => 'Présentation introuvable.'], Response::HTTP_NOT_FOUND);
            }

            $presentation->delete();
            return response()->json(['message' => 'Présentation supprimée avec succès'], Response::HTTP_OK);
        } catch (\Exception $e) {
            Log::error("Erreur lors de la suppression de la présentation : " . $e->getMessage());
            return response()->json(['error' => 'Impossible de supprimer la présentation.'], Response::HTTP_INTERNAL_SERVER_ERROR);
        }
    }
}
