<?php

namespace App\Http\Controllers\API;

use App\Http\Controllers\Controller;
use App\Models\Mission;
use Illuminate\Http\Request;
use Stichoza\GoogleTranslate\GoogleTranslate;
use Illuminate\Support\Facades\Log;

class MissionController extends Controller
{
    /**
     * Récupérer toutes les missions avec la langue sélectionnée
     */
    public function index(Request $request)
    {
        try {
            $lang = $request->header('Accept-Language', 'fr'); // Langue demandée (FR par défaut)

            // Récupérer toutes les missions
            $missions = Mission::all()->map(function ($mission) use ($lang) {
                return [
                    'id' => $mission->id,
                    'titre' => $lang === 'en' ? $mission->titre_en : $mission->titre,
                    'created_at' => $mission->created_at,
                ];
            });

            return response()->json($missions, 200);
        } catch (\Exception $e) {
            Log::error("Erreur lors de la récupération des missions : " . $e->getMessage());
            return response()->json(['error' => 'Une erreur est survenue lors de la récupération des missions.'], 500);
        }
    }

    /**
     * Créer une nouvelle mission avec traduction automatique
     */
    public function store(Request $request)
    {
        try {
            // Validation des données reçues
            $validated = $request->validate([
                'titre' => 'required|string|max:255',
            ]);

            // Traduction automatique en anglais
            $translator = new GoogleTranslate();
            $translator->setSource('fr');
            $translator->setTarget('en');
            $titre_en = $translator->translate($validated['titre']);

            // Création de la mission avec traduction
            $mission = Mission::create([
                'titre' => $validated['titre'],
                'titre_en' => $titre_en
            ]);

            return response()->json($mission, 201);
        } catch (\Exception $e) {
            Log::error("Erreur lors de la création de la mission : " . $e->getMessage());
            return response()->json(['error' => 'Impossible d\'ajouter la mission.'], 500);
        }
    }

    /**
     * Afficher une mission spécifique
     */
    public function show(Request $request, $id)
    {
        try {
            $lang = $request->header('Accept-Language', 'fr'); // Langue demandée

            // Récupérer la mission par ID
            $mission = Mission::find($id);
            if (!$mission) {
                return response()->json(['error' => 'Mission introuvable.'], 404);
            }

            // Retourner la mission dans la langue demandée
            return response()->json([
                'id' => $mission->id,
                'titre' => $lang === 'en' ? $mission->titre_en : $mission->titre,
                'created_at' => $mission->created_at,
            ], 200);
        } catch (\Exception $e) {
            Log::error("Erreur lors de la récupération de la mission : " . $e->getMessage());
            return response()->json(['error' => 'Une erreur est survenue.'], 500);
        }
    }

    /**
     * Mettre à jour une mission existante
     */
    public function update(Request $request, $id)
    {
        try {
            $validated = $request->validate([
                'titre' => 'required|string|max:255',
            ]);

            $mission = Mission::find($id);
            if (!$mission) {
                return response()->json(['error' => 'Mission introuvable.'], 404);
            }

            // Mise à jour et retraduction du titre en anglais
            $translator = new GoogleTranslate();
            $translator->setSource('fr');
            $translator->setTarget('en');
            $titre_en = $translator->translate($validated['titre']);

            $mission->update([
                'titre' => $validated['titre'],
                'titre_en' => $titre_en,
            ]);

            return response()->json($mission, 200);
        } catch (\Exception $e) {
            Log::error("Erreur lors de la mise à jour de la mission : " . $e->getMessage());
            return response()->json(['error' => 'Impossible de mettre à jour la mission.'], 500);
        }
    }

    /**
     * Supprimer une mission
     */
    public function destroy($id)
    {
        try {
            $mission = Mission::find($id);
            if (!$mission) {
                return response()->json(['error' => 'Mission introuvable.'], 404);
            }

            $mission->delete();
            return response()->json(['message' => 'Mission supprimée avec succès.'], 200);
        } catch (\Exception $e) {
            Log::error("Erreur lors de la suppression de la mission : " . $e->getMessage());
            return response()->json(['error' => 'Impossible de supprimer la mission.'], 500);
        }
    }
}
