<?php

namespace App\Http\Controllers\API;

use App\Http\Controllers\Controller;
use Illuminate\Http\Request;
use App\Models\Interview;
use App\Models\InterviewDetail;
use Illuminate\Support\Str;
use Illuminate\Support\Facades\Storage;
use Illuminate\Support\Facades\DB;
use Stichoza\GoogleTranslate\GoogleTranslate;

class InterviewController extends Controller
{
    private const STORAGE_DIRECTORY = 'public/interviews';
    private const MAX_PHOTO_SIZE = 30720; // 30 MB

    public function index(Request $request)
    {
        $lang = $request->header('Accept-Language', 'fr');
        
        $interviews = Interview::with('details')->get()->map(function ($interview) use ($lang) {
            return [
                'id' => $interview->id,
                'titre' => $lang === 'en' ? $interview->titre_en : $interview->titre,
                'slug' => $interview->slug,
                'description' => $lang === 'en' ? $interview->description_en : $interview->description,
                'date' => $interview->date,
                'imagePrincipale' => $interview->imagePrincipale,
                'imagePrincipaleUrl' => $interview->imagePrincipale ? url('/api/interviews/image/' . basename($interview->imagePrincipale)) : null,
                'source' => $interview->source,
                'details' => $interview->details ? [
                    'introduction' => $lang === 'en' ? $interview->details->introduction_en : $interview->details->introduction,
                    'contenu' => $lang === 'en' ? $interview->details->contenu_en : $interview->details->contenu,
                ] : null,
            ];
        });

        return response()->json($interviews);
    }

    public function store(Request $request)
    {
        $request->validate([
            'titre' => 'required|string|unique:interviews,titre',
            'description' => 'required|string',
            'date' => 'required|date',
            'imagePrincipale' => ['required', 'image', 'mimes:jpeg,png,jpg,gif', 'max:' . self::MAX_PHOTO_SIZE],
            'source' => 'required|string',
            'introduction' => 'required|string',
            'contenu' => 'required|string',
        ]);

        DB::beginTransaction();
        try {
            $imagePath = $request->file('imagePrincipale')->store(self::STORAGE_DIRECTORY);
            $translator = new GoogleTranslate();
            $translator->setSource('fr');
            $translator->setTarget('en');

            $interview = Interview::create([
                'titre' => $request->titre,
                'titre_en' => $translator->translate($request->titre),
                'slug' => Str::slug($request->titre),
                'description' => $request->description,
                'description_en' => $translator->translate($request->description),
                'date' => $request->date,
                'imagePrincipale' => $imagePath,
                'source' => $request->source,
            ]);

            InterviewDetail::create([
                'interview_id' => $interview->id,
                'introduction' => $request->introduction,
                'introduction_en' => $translator->translate($request->introduction),
                'contenu' => $request->contenu,
                'contenu_en' => $translator->translate($request->contenu),
            ]);

            DB::commit();
            
            return response()->json($interview->load('details'), 201);
        } catch (\Exception $e) {
            DB::rollback();
            Storage::delete($imagePath ?? '');
            return response()->json(['error' => 'Erreur lors de la création de l\'interview : ' . $e->getMessage()], 500);
        }
    }

    public function show(Request $request, $slug)
    {
        $lang = $request->header('Accept-Language', 'fr');
        
        $interview = Interview::with('details')->where('slug', $slug)->firstOrFail();
        
        $response = [
            'id' => $interview->id,
            'titre' => $lang === 'en' ? $interview->titre_en : $interview->titre,
            'slug' => $interview->slug,
            'description' => $lang === 'en' ? $interview->description_en : $interview->description,
            'date' => $interview->date,
            'imagePrincipale' => $interview->imagePrincipale,
            'imagePrincipaleUrl' => $interview->imagePrincipale ? url('/api/interviews/image/' . basename($interview->imagePrincipale)) : null,
            'source' => $interview->source,
            'details' => $interview->details ? [
                'introduction' => $lang === 'en' ? $interview->details->introduction_en : $interview->details->introduction,
                'contenu' => $lang === 'en' ? $interview->details->contenu_en : $interview->details->contenu,
            ] : null,
        ];

        return response()->json($response);
    }

    public function update(Request $request, $id)
    {
        $interview = Interview::findOrFail($id);
        
        $request->validate([
            'titre' => 'sometimes|string|unique:interviews,titre,' . $id,
            'description' => 'sometimes|string',
            'date' => 'sometimes|date',
            'imagePrincipale' => ['nullable', 'image', 'mimes:jpeg,png,jpg,gif', 'max:' . self::MAX_PHOTO_SIZE],
            'source' => 'sometimes|string',
            'introduction' => 'sometimes|string',
            'contenu' => 'sometimes|string',
        ]);

        DB::beginTransaction();
        try {
            if ($request->hasFile('imagePrincipale')) {
                Storage::delete($interview->imagePrincipale);
                $interview->imagePrincipale = $request->file('imagePrincipale')->store(self::STORAGE_DIRECTORY);
            }

            $translator = new GoogleTranslate();
            $translator->setSource('fr');
            $translator->setTarget('en');

            $updateData = [];
            if ($request->has('titre')) {
                $updateData['titre'] = $request->titre;
                $updateData['titre_en'] = $translator->translate($request->titre);
                $updateData['slug'] = Str::slug($request->titre);
            }
            if ($request->has('description')) {
                $updateData['description'] = $request->description;
                $updateData['description_en'] = $translator->translate($request->description);
            }
            if ($request->has('date')) {
                $updateData['date'] = $request->date;
            }
            if ($request->has('source')) {
                $updateData['source'] = $request->source;
            }

            $interview->update($updateData);

            if ($interview->details) {
                $detailsUpdate = [];
                if ($request->has('introduction')) {
                    $detailsUpdate['introduction'] = $request->introduction;
                    $detailsUpdate['introduction_en'] = $translator->translate($request->introduction);
                }
                if ($request->has('contenu')) {
                    $detailsUpdate['contenu'] = $request->contenu;
                    $detailsUpdate['contenu_en'] = $translator->translate($request->contenu);
                }
                $interview->details->update($detailsUpdate);
            } else {
                InterviewDetail::create([
                    'interview_id' => $interview->id,
                    'introduction' => $request->introduction,
                    'introduction_en' => $translator->translate($request->introduction),
                    'contenu' => $request->contenu,
                    'contenu_en' => $translator->translate($request->contenu),
                ]);
            }

            DB::commit();
            return response()->json($interview->load('details'));
        } catch (\Exception $e) {
            DB::rollback();
            return response()->json(['error' => 'Erreur lors de la mise à jour : ' . $e->getMessage()], 500);
        }
    }
    public function destroy($id)
    {
        $interview = Interview::findOrFail($id);

        DB::beginTransaction();
        try {
            Storage::delete($interview->imagePrincipale);
            $interview->delete();
            DB::commit();
            return response()->json(null, 204);
        } catch (\Exception $e) {
            DB::rollback();
            return response()->json(['error' => 'Erreur lors de la suppression : ' . $e->getMessage()], 500);
        }
    }
}
