<?php
namespace App\Http\Controllers\API;

use App\Http\Controllers\Controller;
use Illuminate\Http\Request;
use Illuminate\Http\Response;
use Illuminate\Support\Facades\Validator;
use App\Models\Domaine;
use Stichoza\GoogleTranslate\GoogleTranslate;

class DomaineController extends Controller
{
    /**
     * Liste tous les domaines avec traduction
     */
    public function index(Request $request)
    {
        $lang = $request->header('Accept-Language', 'fr'); 

        $domaines = Domaine::all()->map(function ($domaine) use ($lang) {
            return [
                'id' => $domaine->id,
                'nom' => $lang === 'en' ? $domaine->nom_en : $domaine->nom,
                'created_at' => $domaine->created_at,
            ];
        });

        return response()->json(['data' => $domaines], Response::HTTP_OK);
    }

    /**
     * Crée un domaine avec traduction automatique
     */
    public function store(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'nom' => 'required|string|max:255',
        ]);

        if ($validator->fails()) {
            return response()->json(['errors' => $validator->errors()], Response::HTTP_UNPROCESSABLE_ENTITY);
        }

        // Traduction automatique
        $translator = new GoogleTranslate();
        $translator->setSource('fr');
        $translator->setTarget('en');
        $nom_en = $translator->translate($request->nom);

        $domaine = Domaine::create([
            'nom' => $request->nom,
            'nom_en' => $nom_en,
        ]);

        return response()->json(['data' => $domaine, 'message' => 'Domaine créé avec succès'], Response::HTTP_CREATED);
    }

    /**
     * Affiche un domaine avec traduction
     */
    public function show(Request $request, Domaine $domaine)
    {
        $lang = $request->header('Accept-Language', 'fr');

        return response()->json([
            'id' => $domaine->id,
            'nom' => $lang === 'en' ? $domaine->nom_en : $domaine->nom,
            'created_at' => $domaine->created_at,
        ], Response::HTTP_OK);
    }

    /**
     * Met à jour un domaine avec traduction
     */
    public function update(Request $request, Domaine $domaine)
    {
        $validator = Validator::make($request->all(), [
            'nom' => 'required|string|max:255',
        ]);

        if ($validator->fails()) {
            return response()->json(['errors' => $validator->errors()], Response::HTTP_UNPROCESSABLE_ENTITY);
        }

        // Traduction automatique
        $translator = new GoogleTranslate();
        $translator->setSource('fr');
        $translator->setTarget('en');
        $nom_en = $translator->translate($request->nom);

        $domaine->update([
            'nom' => $request->nom,
            'nom_en' => $nom_en,
        ]);

        return response()->json(['data' => $domaine, 'message' => 'Domaine mis à jour avec succès'], Response::HTTP_OK);
    }

    /**
     * Supprime un domaine
     */
    public function destroy(Domaine $domaine)
    {
        $domaine->delete();
        return response()->json(['message' => 'Domaine supprimé avec succès'], Response::HTTP_OK);
    }
}
