<?php

namespace App\Http\Controllers\API;

use App\Http\Controllers\Controller;
use Illuminate\Http\Request;
use App\Models\Content;

use Illuminate\Support\Facades\Storage;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Validator;

class ContentController extends Controller
{
    /**
     * Liste tous les contenus.
     *
     * @return \Illuminate\Http\JsonResponse
     */
    public function index()
    {
        $contents = Content::all();

        return $this->successResponse(
            'Liste de tous les contenus récupérée avec succès.',
            $contents
        );
    }

    /**
     * Créer un nouveau contenu.
     *
     * @param  \Illuminate\Http\Request  $request
     * @return \Illuminate\Http\JsonResponse
     */
    public function store(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'titre' => 'required|string|max:255',
            'Description' => 'required|string',
            'image' => 'nullable|image|mimes:jpeg,png,jpg,gif|max:2048',
            'video' => 'nullable|url',
            'type' => 'required|in:actualite,publication,communique,interview,mediatheque',
        ]);

        if ($validator->fails()) {
            return $this->errorResponse('Erreur de validation', $validator->errors(), 422);
        }

        $imagePath = null;
        if ($request->hasFile('image')) {
            $imagePath = $request->file('image')->store('images', 'public');
        }

        $content = Content::create([
            'titre' => $request->titre,
            'Description' => $request->Description,
            'image' => $imagePath,
            'video' => $request->video,
            'type' => $request->type,
            'admin_id' => Auth::id(),
        ]);

        return $this->successResponse('Contenu créé avec succès.', $content, 201);
    }

    /**
     * Récupère un contenu par son ID.
     *
     * @param  int  $id
     * @return \Illuminate\Http\JsonResponse
     */
    public function show($id)
    {
        $content = Content::find($id);

        if (!$content) {
            return $this->errorResponse('Contenu non trouvé.', null, 404);
        }

        return $this->successResponse('Contenu récupéré avec succès.', $content);
    }

    /**
     * Met à jour un contenu existant.
     *
     * @param  \Illuminate\Http\Request  $request
     * @param  int  $id
     * @return \Illuminate\Http\JsonResponse
     */
    public function update(Request $request, $id)
    {
        $content = Content::find($id);

        if (!$content) {
            return $this->errorResponse('Contenu non trouvé.', null, 404);
        }

        $validator = Validator::make($request->all(), [
            'titre' => 'sometimes|required|string|max:255',
            'Description' => 'sometimes|required|string',
            'image' => 'nullable|image|mimes:jpeg,png,jpg,gif|max:2048',
            'video' => 'nullable|url',
            'type' => 'sometimes|required|in:actualite,publication,communique,interview,mediatheque',
        ]);

        if ($validator->fails()) {
           return $this->errorResponse('Erreur de validation', $validator->errors(), 422);
        }


        if ($request->hasFile('image')) {
             if ($content->image) {
                Storage::disk('public')->delete($content->image);
            }
            $content->image = $request->file('image')->store('images', 'public');
        }


        $content->fill($request->only(['titre', 'Description', 'video', 'type']));
         $content->save();



        return $this->successResponse('Contenu mis à jour avec succès.', $content);
    }

    /**
     * Supprime un contenu.
     *
     * @param  int  $id
     * @return \Illuminate\Http\JsonResponse
     */
    public function destroy($id)
    {
        $content = Content::find($id);

        if (!$content) {
            return $this->errorResponse('Contenu non trouvé.', null, 404);
        }

        if ($content->image) {
            Storage::disk('public')->delete($content->image);
        }

        $content->delete();

       return $this->successResponse('Contenu supprimé avec succès.', null, 204);

    }

    /**
     * Récupère les contenus par type (actualite, publication, etc).
     *
     * @param string $type
     * @return \Illuminate\Http\JsonResponse
     */
    public function getContentsByType($type)
    {
        $allowedTypes = ['actualite', 'publication', 'communique', 'interview', 'mediatheque'];

        if (!in_array($type, $allowedTypes)) {
            return $this->errorResponse('Type de contenu invalide.', null, 400);
        }

        $contents = Content::where('type', $type)->get();

        return $this->successResponse(
            "Contenus de type \"{$type}\" récupérés avec succès.",
            $contents
        );
    }

    /**
     * Structure une réponse JSON de succès.
     *
     * @param  string  $message
     * @param  mixed  $data
     * @param  int  $statusCode
     * @return \Illuminate\Http\JsonResponse
     */
    protected function successResponse($message, $data = null, $statusCode = 200)
    {
        return response()->json([
            'code' => $statusCode,
            'message' => $message,
            'data' => $data,
        ], $statusCode);
    }


    /**
    * Structure une réponse JSON d'erreur.
     *
     * @param string $message
     * @param mixed $errors
     * @param int $statusCode
     * @return \Illuminate\Http\JsonResponse
     */

    protected function errorResponse($message, $errors = null, $statusCode = 400)
     {
        return response()->json([
            'code' => $statusCode,
            'message' => $message,
            'errors' => $errors,
        ], $statusCode);
    }
}
