<?php

namespace App\Http\Controllers\API;

use App\Http\Controllers\Controller;
use Illuminate\Http\Request;
use App\Models\Communication;
use Illuminate\Support\Facades\Storage;
use Stichoza\GoogleTranslate\GoogleTranslate;

class CommunicationController extends Controller
{
    private const PHOTO_STORAGE_PATH = 'public/photos';
    private const DOCUMENT_STORAGE_PATH = 'public/documents';

    private const MAX_PHOTO_SIZE = 30720; // 30 MB
    private const MAX_DOCUMENT_SIZE = 25600; // 25 MB

    /**
     * Display a listing of the resource.
     */
    public function index(Request $request)
    {
        $lang = $request->header('Accept-Language', 'fr'); // Langue demandée (FR par défaut)

        $communications = Communication::all()->map(function ($communication) use ($lang) {
            $communication->title = $lang === 'en' ? $communication->title_en : $communication->title;
            $communication->description = $lang === 'en' ? $communication->description_en : $communication->description;
            $communication->photo = $communication->photo ? url('/api/images_communication/' . basename($communication->photo)) : null;
            $communication->document = $communication->document ? url('/api/documents_communication/' . basename($communication->document)) : null;
            return $communication;
        });

        return response()->json($communications);
    }

    /**
     * Store a newly created resource in storage.
     */
    public function store(Request $request)
    {
        $request->validate([
            'title' => 'required|string',
            'description' => 'nullable|string',
            'photo' => ['nullable', 'image', 'mimes:jpeg,png,jpg,gif', 'max:' . self::MAX_PHOTO_SIZE],
            'document' => ['required', 'file', 'mimes:pdf,doc,docx', 'max:' . self::MAX_DOCUMENT_SIZE],
            'communication_date' => 'nullable|date',
        ]);

        $documentPath = $request->hasFile('document') ? $request->file('document')->store(self::DOCUMENT_STORAGE_PATH) : null;
        $photoPath = $request->hasFile('photo') ? $request->file('photo')->store(self::PHOTO_STORAGE_PATH) : null;

        $communication = Communication::create([
            'title' => $request->input('title'),
            'description' => $request->input('description'),
            'photo' => $photoPath,
            'document' => $documentPath,
            'communication_date' => $request->input('communication_date'),
        ]);

        $communication->photo = $communication->photo ? url('/api/images_communication/' . basename($communication->photo)) : null;
        $communication->document = $communication->document ? url('/api/documents_communication/' . basename($communication->document)) : null;

        return response()->json($communication, 201);
    }

    /**
     * Display the specified resource.
     */
    public function show(Request $request, Communication $communication)
    {
        $lang = $request->header('Accept-Language', 'fr'); // Langue demandée

        $communication->title = $lang === 'en' ? $communication->title_en : $communication->title;
        $communication->description = $lang === 'en' ? $communication->description_en : $communication->description;
        $communication->photo = $communication->photo ? url('/api/images_communication/' . basename($communication->photo)) : null;
        $communication->document = $communication->document ? url('/api/documents_communication/' . basename($communication->document)) : null;

        return response()->json($communication);
    }

    /**
     * Update the specified resource in storage.
     */
    public function update(Request $request, Communication $communication)
    {
        $request->validate([
            'title' => 'sometimes|required|string',
            'description' => 'nullable|string',
            'photo' => ['nullable', 'image', 'mimes:jpeg,png,jpg,gif', 'max:' . self::MAX_PHOTO_SIZE],
            'document' => ['nullable', 'file', 'mimes:pdf,doc,docx', 'max:' . self::MAX_DOCUMENT_SIZE],
            'communication_date' => 'nullable|date',
        ]);

        if ($request->has('title')) {
            $communication->title = $request->input('title');
        }

        if ($request->has('description')) {
            $communication->description = $request->input('description');
        }

        if ($request->has('communication_date')) {
            $communication->communication_date = $request->input('communication_date');
        }

        if ($request->hasFile('document')) {
            if ($communication->document) {
                Storage::delete($communication->document);
            }
            $communication->document = $request->file('document')->store(self::DOCUMENT_STORAGE_PATH);
        }

        if ($request->hasFile('photo')) {
            if ($communication->photo) {
                Storage::delete($communication->photo);
            }
            $communication->photo = $request->file('photo')->store(self::PHOTO_STORAGE_PATH);
        }

        $communication->save();

        $communication->photo = $communication->photo ? url('/api/images_communication/' . basename($communication->photo)) : null;
        $communication->document = $communication->document ? url('/api/documents_communication/' . basename($communication->document)) : null;

        return response()->json($communication);
    }

    /**
     * Remove the specified resource from storage.
     */
    public function destroy(Communication $communication)
    {
        if ($communication->document) {
            Storage::delete($communication->document);
        }
        if ($communication->photo) {
            Storage::delete($communication->photo);
        }

        $communication->delete();

        return response()->json(null, 204);
    }
}
