<?php

namespace App\Http\Controllers\API;

use App\Http\Controllers\Controller;
use Illuminate\Http\Request;
use App\Models\Actualite;
use App\Models\ActualiteDetail;
use Illuminate\Support\Str;
use Illuminate\Support\Facades\Storage;
use Illuminate\Support\Facades\Validator;
use Stichoza\GoogleTranslate\GoogleTranslate; // Added
use Exception;

class ActualiteController extends Controller
{
    const MAX_IMAGE_SIZE = 9048000000000000000000000000000; // 9MB
    const IMAGE_PATH = 'images_actualite';

    /**
     * Stores an image and returns its path.
     */
    private function storeImage($image)
    {
        if (!$image) return null;

        // Generate a unique name for the image
        $fileName = time() . '_' . Str::random(10) . '.' . $image->getClientOriginalExtension();

        // Store the image in the public folder
        $image->move(public_path(self::IMAGE_PATH), $fileName);

        // Return the relative path
        return self::IMAGE_PATH . '/' . $fileName;
    }

    /**
     * Deletes an image.
     */
    private function deleteImage($path)
    {
        if ($path && file_exists(public_path($path))) {
            unlink(public_path($path));
        }
    }

    /**
     * Stores multiple images and returns their paths.
     */
    private function storeGalleryPhotos($photos)
    {
        if (!$photos) return [];

        $paths = [];
        foreach ($photos as $photo) {
            $path = $this->storeImage($photo);
            if ($path) {
                $paths[] = $path;
            }
        }
        return $paths;
    }

    /**
     * Formats the image URL for the API.
     */
    private function formatImageUrl($path)
    {
        return $path ? url('/' . $path) : null; // Fixed: removed 'api' prefix
    }

    /**
     * Lists all news articles.
     */
    public function index(Request $request)
    {
        $lang = $request->header('Accept-Language', 'fr');

        $actualites = Actualite::with('details')->get()->map(function ($actualite) use ($lang) {
            $actualite->titre = $lang === 'en' ? $actualite->titre_en : $actualite->titre;
            $actualite->description = $lang === 'en' ? $actualite->description_en : $actualite->description;
            $actualite->image = $this->formatImageUrl($actualite->image);

            if ($actualite->details) {
                $actualite->details->imagePrincipale = $this->formatImageUrl($actualite->details->imagePrincipale);

                if ($actualite->details->galleryPhotos) {
                    $actualite->details->galleryPhotos = array_map(
                        fn ($photo) => $this->formatImageUrl($photo),
                        is_array($actualite->details->galleryPhotos)
                            ? $actualite->details->galleryPhotos
                            : json_decode($actualite->details->galleryPhotos, true) ?? []
                    );
                }
            }

            return $actualite;
        });

        return response()->json($actualites);
    }


    /**
     * Creates a new news article.
     */
    public function store(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'titre' => 'required|string|max:255|unique:actualites',
            'description' => 'required|string',
            'image' => 'nullable|image|mimes:jpeg,png,jpg,gif|max:' . self::MAX_IMAGE_SIZE,
            'datePublication' => 'nullable|date',
            'imagePrincipale' => 'nullable|image|mimes:jpeg,png,jpg,gif|max:' . self::MAX_IMAGE_SIZE,
            'galleryPhotos' => 'nullable|array',
            'galleryPhotos.*' => 'image|mimes:jpeg,png,jpg,gif|max:' . self::MAX_IMAGE_SIZE,
        ]);

        if ($validator->fails()) {
            return response()->json(['status' => false, 'errors' => $validator->errors()], 422);
        }

        try {
            // Create the news article
            $actualiteData = $request->except(['image', 'imagePrincipale', 'galleryPhotos']);

             // Auto-translate titre and description to English
            $translator = new GoogleTranslate();
            $translator->setSource('fr');
            $translator->setTarget('en');

            $actualiteData['titre_en'] = $translator->translate($request->titre);
            $actualiteData['description_en'] = $translator->translate($request->description);

            // Manage the main image of the news article
            if ($request->hasFile('image')) {
                $actualiteData['image'] = $this->storeImage($request->file('image'));
            }

            $actualite = Actualite::create($actualiteData);

            // Manage details and images
            $detailData = [];

            if ($request->hasFile('imagePrincipale')) {
                $detailData['imagePrincipale'] = $this->storeImage($request->file('imagePrincipale'));
            }

            if ($request->hasFile('galleryPhotos')) {
                $detailData['galleryPhotos'] = $this->storeGalleryPhotos($request->file('galleryPhotos'));
            }

            $actualite->details()->create($detailData);

            // Reload the news article with its details and format the image URLs
            $actualite->load('details');
            $actualite->image = $this->formatImageUrl($actualite->image);

            if ($actualite->details) {
                $actualite->details->imagePrincipale = $this->formatImageUrl($actualite->details->imagePrincipale);
                $actualite->details->galleryPhotos = array_map(
                    fn ($photo) => $this->formatImageUrl($photo),
                    $actualite->details->galleryPhotos ?? []
                );
            }

            return response()->json([
                'status' => true,
                'message' => 'Actualité créée avec succès',
                'data' => $actualite
            ], 201);
        } catch (Exception $e) { // Corrected to Exception
            return response()->json([
                'status' => false,
                'message' => 'Erreur: ' . $e->getMessage()
            ], 500);
        }
    }

    /**
     * Displays a specific news article.
     */
    public function show(Request $request, $slug)
    {
        $lang = $request->header('Accept-Language', 'fr');

        $actualite = Actualite::where('slug', $slug)->with('details')->first();

        if (!$actualite) {
            return response()->json(['message' => 'Actualité non trouvée'], 404);
        }

        $actualite->titre = $lang === 'en' ? $actualite->titre_en : $actualite->titre;
        $actualite->description = $lang === 'en' ? $actualite->description_en : $actualite->description;
        $actualite->image = $this->formatImageUrl($actualite->image);

        if ($actualite->details) {
            $actualite->details->imagePrincipale = $this->formatImageUrl($actualite->details->imagePrincipale);
            if ($actualite->details->galleryPhotos) {

                $actualite->details->galleryPhotos = array_map(fn ($photo) => $this->formatImageUrl($photo), is_array($actualite->details->galleryPhotos)
                    ? $actualite->details->galleryPhotos : json_decode($actualite->details->galleryPhotos, true) ?? []);
            }
        }

        return response()->json($actualite);
    }

    /**
     * Updates a news article.
     */
    public function update(Request $request, $id)
    {
        $actualite = Actualite::find($id);

        if (!$actualite) {
            return response()->json(['message' => 'Actualité non trouvée'], 404);
        }

        $validator = Validator::make($request->all(), [
            'titre' => 'sometimes|string|max:255|unique:actualites,titre,' . $actualite->id,
            'description' => 'sometimes|string',
            'image' => 'nullable|image|mimes:jpeg,png,jpg,gif|max:' . self::MAX_IMAGE_SIZE,
            'datePublication' => 'nullable|date',
            'imagePrincipale' => 'nullable|image|mimes:jpeg,png,jpg,gif|max:' . self::MAX_IMAGE_SIZE,
            'galleryPhotos' => 'nullable|array',
            'galleryPhotos.*' => 'image|mimes:jpeg,png,jpg,gif|max:' . self::MAX_IMAGE_SIZE,
        ]);

        if ($validator->fails()) {
            return response()->json(['status' => false, 'errors' => $validator->errors()], 422);
        }

        try {
            $actualiteData = $request->except(['image', 'imagePrincipale', 'galleryPhotos']);

            // Auto-translate titre and description to English if updated
            $translator = new GoogleTranslate();
            $translator->setSource('fr');
            $translator->setTarget('en');
              if ($request->has('titre')) {
                $actualiteData['titre_en'] = $translator->translate($request->titre);
              }
              if ($request->has('description')) {
                $actualiteData['description_en'] = $translator->translate($request->description);
              }

            // Manage the main image of the news article
            if ($request->hasFile('image')) {
                $this->deleteImage($actualite->image);
                $actualiteData['image'] = $this->storeImage($request->file('image'));
            }

            $actualite->update($actualiteData);

            // Manage details and images
            $detailData = [];

            if ($request->hasFile('imagePrincipale')) {
                if ($actualite->details) {
                    $this->deleteImage($actualite->details->imagePrincipale);
                }
                $detailData['imagePrincipale'] = $this->storeImage($request->file('imagePrincipale'));
            }

            if ($request->hasFile('galleryPhotos')) {
                if ($actualite->details && $actualite->details->galleryPhotos) {
                    foreach ($actualite->details->galleryPhotos as $photo) {
                        $this->deleteImage($photo);
                    }
                }
                $detailData['galleryPhotos'] = $this->storeGalleryPhotos($request->file('galleryPhotos'));
            }

            if ($actualite->details) {
                $actualite->details->update($detailData);
            } else {
                $actualite->details()->create($detailData);
            }

            // Reload and format
            $actualite->load('details');
            $actualite->image = $this->formatImageUrl($actualite->image);

            if ($actualite->details) {
                $actualite->details->imagePrincipale = $this->formatImageUrl($actualite->details->imagePrincipale);
                $actualite->details->galleryPhotos = array_map(
                    fn ($photo) => $this->formatImageUrl($photo),
                    $actualite->details->galleryPhotos ?? []
                );
            }

            return response()->json([
                'status' => true,
                'message' => 'Actualité mise à jour avec succès',
                'data' => $actualite
            ], 200);
        } catch (Exception $e) {
            return response()->json([
                'status' => false,
                'message' => 'Erreur: ' . $e->getMessage()
            ], 500);
        }
    }

    /**
     * Deletes a news article.
     */
    public function destroy($id)
    {
        $actualite = Actualite::find($id);

        if (!$actualite) {
            return response()->json(['message' => 'Actualité non trouvée'], 404);
        }

        try {
            // Delete all images

            $this->deleteImage($actualite->image);

            if ($actualite->details) {
                $this->deleteImage($actualite->details->imagePrincipale);
                if ($actualite->details->galleryPhotos) {
                    foreach ($actualite->details->galleryPhotos as $photo) {
                        $this->deleteImage($photo);
                    }
                }
                $actualite->details->delete();
            }

            $actualite->delete();

            return response()->json([
                'status' => true,
                'message' => 'Actualité supprimée avec succès'
            ], 200);
        } catch (Exception $e) {
            return response()->json([
                'status' => false,
                'message' => 'Erreur: ' . $e->getMessage()
            ], 500);
        }
    }
}